"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = __importDefault(require("express"));
const cors_1 = __importDefault(require("cors"));
const helmet_1 = __importDefault(require("helmet"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const env_1 = require("./config/env");
const db_1 = require("./config/db");
const auth_routes_1 = __importDefault(require("./routes/auth.routes"));
const users_routes_1 = __importDefault(require("./routes/users.routes"));
const leads_routes_1 = __importDefault(require("./routes/leads.routes"));
const companies_routes_1 = __importDefault(require("./routes/companies.routes"));
const tasks_routes_1 = __importDefault(require("./routes/tasks.routes"));
const services_routes_1 = __importDefault(require("./routes/services.routes"));
const notifications_routes_1 = __importDefault(require("./routes/notifications.routes"));
const reports_routes_1 = __importDefault(require("./routes/reports.routes"));
const automation_routes_1 = __importDefault(require("./routes/automation.routes"));
const errorHandler_1 = require("./middleware/errorHandler");
const app = (0, express_1.default)();
app.use((0, helmet_1.default)());
app.use(express_1.default.json());
app.use((0, cors_1.default)({ origin: env_1.env.corsOrigin.length ? env_1.env.corsOrigin : true }));
app.use('/uploads', express_1.default.static(path_1.default.join(__dirname, '../uploads')));
// Disable caching for API routes (prevents stale GET responses behind cPanel/proxies)
app.use(env_1.env.apiBasePath, (_req, res, next) => {
    res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, proxy-revalidate');
    res.setHeader('Pragma', 'no-cache');
    res.setHeader('Expires', '0');
    res.setHeader('Surrogate-Control', 'no-store');
    next();
});
// Build/deploy visibility (set BUILD_VERSION/BUILD_SHA on cPanel if desired)
const BUILD_VERSION = process.env.BUILD_VERSION || process.env.BUILD_SHA || 'dev';
const BUILD_TIME = process.env.BUILD_TIME || new Date().toISOString();
const api = express_1.default.Router();
api.get('/version', (_req, res) => {
    res.setHeader('Cache-Control', 'no-store');
    res.status(200).json({ version: BUILD_VERSION, builtAt: BUILD_TIME });
});
api.use('/auth', auth_routes_1.default);
api.use('/users', users_routes_1.default);
api.use('/leads', leads_routes_1.default);
api.use('/companies', companies_routes_1.default);
api.use('/tasks', tasks_routes_1.default);
api.use('/services', services_routes_1.default);
api.use('/notifications', notifications_routes_1.default);
api.use('/reports', reports_routes_1.default);
api.use('/automation', automation_routes_1.default);
app.use(env_1.env.apiBasePath, api);
// Explicit 404 for unknown API routes (prevents SPA fallback from swallowing /api URLs)
app.use(env_1.env.apiBasePath, (_req, res) => {
    res.status(404).json({ message: 'Not Found' });
});
// Serve the built frontend (SPA) when available.
// This enables a single-domain deployment (e.g. https://system.cybilsolutions.com/ with /api).
// cPanel/Passenger may set process.cwd() differently, so we probe a few common locations.
const frontendDistCandidates = [
    env_1.env.frontendDistDir ? path_1.default.resolve(process.cwd(), env_1.env.frontendDistDir) : null,
    // Repo layout: Backend/ and Frontend/ are siblings
    path_1.default.resolve(process.cwd(), '../Frontend/dist'),
    // When compiled, __dirname is typically Backend/dist
    path_1.default.resolve(__dirname, '../../Frontend/dist'),
    // When running TS directly, __dirname is typically Backend/src
    path_1.default.resolve(__dirname, '../Frontend/dist'),
].filter(Boolean);
const frontendDistDir = frontendDistCandidates.find(p => fs_1.default.existsSync(p));
if (frontendDistDir) {
    const staticMiddleware = express_1.default.static(frontendDistDir, {
        setHeaders: (res, filePath) => {
            // Never cache the SPA shell; allow long-cache for hashed assets
            if (filePath.endsWith(`${path_1.default.sep}index.html`)) {
                res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, proxy-revalidate');
                res.setHeader('Pragma', 'no-cache');
                res.setHeader('Expires', '0');
                return;
            }
            res.setHeader('Cache-Control', 'public, max-age=31536000, immutable');
        },
    });
    app.use((req, res, next) => {
        if (req.path.startsWith(env_1.env.apiBasePath) || req.path.startsWith('/uploads'))
            return next();
        return staticMiddleware(req, res, next);
    });
    app.get('*', (req, res, next) => {
        if (req.path.startsWith(env_1.env.apiBasePath) || req.path.startsWith('/uploads'))
            return next();
        res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, proxy-revalidate');
        res.setHeader('Pragma', 'no-cache');
        res.setHeader('Expires', '0');
        res.sendFile(path_1.default.join(frontendDistDir, 'index.html'));
    });
}
app.use((req, _res, next) => {
    req.basePath = env_1.env.apiBasePath;
    next();
});
app.use(errorHandler_1.errorHandler);
(0, db_1.pingDB)().then(() => {
    app.listen(env_1.env.port, () => {
        console.log(`Cybil CRM backend listening on port ${env_1.env.port} at base ${env_1.env.apiBasePath}`);
    });
}).catch(err => {
    console.error('DB connection failed:', err);
    process.exit(1);
});
