"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const authMiddleware_1 = require("../middleware/authMiddleware");
const tasksRepo_1 = require("../repositories/tasksRepo");
const router = (0, express_1.Router)();
router.get('/', authMiddleware_1.requireAuth, async (req, res) => {
    const tasks = await (0, tasksRepo_1.listTasks)(req.query);
    res.json(tasks.map(t => ({
        id: String(t.uuid),
        title: t.title,
        dueDate: t.due_date || '',
        priority: t.priority,
        status: t.status,
        relatedEntity: t.related_entity_display || '',
        type: t.type || 'Follow-up',
        owner: t.owner_name || '',
        ownerId: t.owner_uuid || undefined,
        description: t.description || undefined,
        relatedCompanyId: t.related_company_id || undefined,
        relatedLeadId: t.related_lead_id || undefined,
    })));
});
router.get('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    const t = await (0, tasksRepo_1.getTaskByUuid)(req.params.uuid);
    if (!t)
        return res.status(404).json({ message: 'Task not found' });
    res.json({
        id: String(t.uuid),
        title: t.title,
        dueDate: t.due_date || '',
        priority: t.priority,
        status: t.status,
        relatedEntity: t.related_entity_display || '',
        type: t.type || 'Follow-up',
        owner: t.owner_name || '',
        ownerId: t.owner_uuid || undefined,
        description: t.description || undefined,
        relatedCompanyId: t.related_company_id || undefined,
        relatedLeadId: t.related_lead_id || undefined,
    });
});
router.post('/', authMiddleware_1.requireAuth, async (req, res) => {
    try {
        const id = await (0, tasksRepo_1.createTask)(req.body);
        res.status(201).json({ id });
    }
    catch (err) {
        console.error('Error creating task:', err);
        res.status(500).json({ message: 'Failed to create task', error: err.message });
    }
});
router.put('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    try {
        const ok = await (0, tasksRepo_1.updateTask)(req.params.uuid, req.body);
        if (!ok)
            return res.status(404).json({ message: 'Not found' });
        res.json({ ok: true });
    }
    catch (err) {
        console.error('Error updating task:', err);
        res.status(500).json({ message: 'Failed to update task', error: err.message });
    }
});
router.patch('/:uuid/status', authMiddleware_1.requireAuth, async (req, res) => {
    try {
        const ok = await (0, tasksRepo_1.patchTaskStatus)(req.params.uuid, req.body.status);
        if (!ok)
            return res.status(404).json({ message: 'Not found' });
        res.json({ ok: true });
    }
    catch (err) {
        console.error('Error updating task status:', err);
        res.status(500).json({ message: 'Failed to update task status', error: err.message });
    }
});
router.delete('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    try {
        const ok = await (0, tasksRepo_1.deleteTask)(req.params.uuid);
        if (!ok)
            return res.status(404).json({ message: 'Not found' });
        res.json({ ok: true });
    }
    catch (err) {
        console.error('Error deleting task:', err);
        res.status(500).json({ message: 'Failed to delete task', error: err.message });
    }
});
exports.default = router;
