"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const authMiddleware_1 = require("../middleware/authMiddleware");
const servicesRepo_1 = require("../repositories/servicesRepo");
const router = (0, express_1.Router)();
router.get('/', authMiddleware_1.requireAuth, async (_req, res) => {
    const rows = await (0, servicesRepo_1.listServices)();
    const services = await Promise.all(rows.map(async (s) => {
        const rules = await (0, servicesRepo_1.listRules)(s.uuid);
        return {
            id: String(s.uuid),
            name: s.name,
            description: s.description || '',
            active: !!s.active,
            checklist: Array.isArray(s.checklist_json) ? s.checklist_json : [],
            rules: rules ? rules.map(r => ({ id: String(r.id), type: r.rule_type, triggerDays: r.trigger_days, description: r.description || '' })) : []
        };
    }));
    res.json(services);
});
router.get('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    const data = await (0, servicesRepo_1.getServiceByUuid)(req.params.uuid);
    if (!data)
        return res.status(404).json({ message: 'Service not found' });
    const s = data.service;
    const rules = data.rules.map(r => ({ id: String(r.id), type: r.rule_type, triggerDays: r.trigger_days, description: r.description || '' }));
    res.json({ id: String(s.uuid), name: s.name, description: s.description || '', active: !!s.active, checklist: Array.isArray(s.checklist_json) ? s.checklist_json : [], rules });
});
router.post('/', authMiddleware_1.requireAuth, async (req, res) => {
    const id = await (0, servicesRepo_1.createService)(req.body);
    res.status(201).json({ id });
});
router.put('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    const ok = await (0, servicesRepo_1.updateService)(req.params.uuid, req.body);
    if (!ok)
        return res.status(404).json({ message: 'Not found' });
    res.json({ ok: true });
});
router.delete('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    const ok = await (0, servicesRepo_1.deleteService)(req.params.uuid);
    if (!ok)
        return res.status(404).json({ message: 'Not found' });
    res.json({ ok: true });
});
router.get('/:uuid/rules', authMiddleware_1.requireAuth, async (req, res) => {
    const rules = await (0, servicesRepo_1.listRules)(req.params.uuid);
    if (rules === null)
        return res.status(404).json({ message: 'Service not found' });
    res.json(rules.map(r => ({ id: String(r.id), type: r.rule_type, triggerDays: r.trigger_days, description: r.description || '' })));
});
router.post('/:uuid/rules', authMiddleware_1.requireAuth, async (req, res) => {
    const id = await (0, servicesRepo_1.addRule)(req.params.uuid, req.body);
    if (!id)
        return res.status(404).json({ message: 'Service not found' });
    res.status(201).json({ id });
});
exports.default = router;
