"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const authMiddleware_1 = require("../middleware/authMiddleware");
const leadsRepo_1 = require("../repositories/leadsRepo");
const router = (0, express_1.Router)();
function parseJsonArray(value) {
    if (Array.isArray(value))
        return value;
    if (value == null)
        return [];
    try {
        const asString = Buffer.isBuffer(value) ? value.toString('utf8') : String(value);
        const parsed = JSON.parse(asString);
        return Array.isArray(parsed) ? parsed : [];
    }
    catch {
        return [];
    }
}
router.get('/', authMiddleware_1.requireAuth, async (req, res) => {
    const r = req;
    const leads = await (0, leadsRepo_1.listLeads)(req.query, r.user);
    res.json(leads.map(l => ({
        id: String(l.uuid),
        name: l.name,
        companyName: l.company_name,
        email: l.email,
        phone: l.phone,
        service: l.service,
        status: l.status,
        owner: l.owner_name || '',
        source: l.source,
        preferredContactMethod: l.preferred_contact_method,
        value: Number(l.value || 0),
        notes: l.notes || undefined,
        tags: parseJsonArray(l.tags_json),
        companyOpenDate: l.company_open_date || undefined,
        createdAt: l.created_at,
    })));
});
router.get('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    const data = await (0, leadsRepo_1.getLeadByUuid)(req.params.uuid);
    if (!data)
        return res.status(404).json({ message: 'Lead not found' });
    const l = data.lead;
    const activities = data.activities.map(a => ({
        id: String(a.id),
        type: a.type,
        content: a.content,
        date: a.activity_date,
        user: a.user_name || '',
    }));
    res.json({
        id: String(l.uuid),
        name: l.name,
        companyName: l.company_name,
        email: l.email,
        phone: l.phone,
        service: l.service,
        status: l.status,
        owner: l.owner_name || '',
        source: l.source,
        preferredContactMethod: l.preferred_contact_method,
        value: Number(l.value || 0),
        notes: l.notes || undefined,
        tags: parseJsonArray(l.tags_json),
        companyOpenDate: l.company_open_date || undefined,
        createdAt: l.created_at,
        activities,
    });
});
router.post('/', authMiddleware_1.requireAuth, async (req, res) => {
    const r = req;
    try {
        const result = await (0, leadsRepo_1.createLead)(req.body, r.user.id);
        res.json(result);
    }
    catch (err) {
        console.error(err);
        res.status(500).json({ message: 'Failed to create lead' });
    }
});
router.put('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    const r = req;
    try {
        await (0, leadsRepo_1.updateLead)(req.params.uuid, req.body);
        if (req.body.activity) {
            await (0, leadsRepo_1.addLeadActivity)(req.params.uuid, req.body.activity, r.user.id);
        }
        res.json({ ok: true });
    }
    catch (err) {
        console.error(err);
        res.status(500).json({ message: 'Failed to update lead' });
    }
});
exports.default = router;
