"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const authMiddleware_1 = require("../middleware/authMiddleware");
const upload_1 = require("../middleware/upload");
const companiesRepo_1 = require("../repositories/companiesRepo");
const router = (0, express_1.Router)();
function parseJsonArray(value) {
    if (Array.isArray(value))
        return value;
    if (value == null)
        return [];
    try {
        const asString = Buffer.isBuffer(value) ? value.toString('utf8') : String(value);
        const parsed = JSON.parse(asString);
        return Array.isArray(parsed) ? parsed : [];
    }
    catch {
        return [];
    }
}
router.get('/', authMiddleware_1.requireAuth, async (req, res) => {
    const companies = await (0, companiesRepo_1.listCompanies)(req.query);
    res.json(companies.map(c => ({
        id: String(c.uuid),
        name: c.name,
        tradeLicense: c.trade_license,
        incorporationDate: c.incorporation_date,
        licenseIssuedDate: c.license_issued_date || undefined,
        lastRenewalDate: c.last_renewal_date || undefined,
        renewalDate: c.renewal_date,
        status: c.status,
        isLoanEligible: !!c.is_loan_eligible,
        vatNumber: c.vat_number || undefined,
        vatRegistrationDate: c.vat_registration_date || undefined,
        bankName: c.bank_name || undefined,
        activeProducts: parseJsonArray(c.active_products_json),
        linkedLeadId: c.linked_lead_id || undefined,
        contacts: c.contacts ? c.contacts.map((ct) => ({ id: String(ct.id), name: ct.name, role: ct.role, email: ct.email, phone: ct.phone, isPrimary: !!ct.is_primary })) : [],
        documents: c.documents ? c.documents.map((d) => ({ id: String(d.id), name: d.name, url: d.url, type: d.type, uploadedAt: d.uploaded_at })) : [],
    })));
});
router.get('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    const data = await (0, companiesRepo_1.getCompanyByUuid)(req.params.uuid);
    if (!data)
        return res.status(404).json({ message: 'Company not found' });
    const c = data.company;
    const contacts = data.contacts.map(ct => ({ id: String(ct.id), name: ct.name, role: ct.role, email: ct.email, phone: ct.phone, isPrimary: !!ct.is_primary }));
    const comm = data.preferences ? { email: !!data.preferences.email, sms: !!data.preferences.sms, whatsapp: !!data.preferences.whatsapp } : { email: true, sms: false, whatsapp: true };
    const documents = data.documents.map(d => ({ id: String(d.id), name: d.name, url: d.url, type: d.type, uploadedAt: d.uploaded_at }));
    res.json({
        id: String(c.uuid),
        name: c.name,
        tradeLicense: c.trade_license,
        incorporationDate: c.incorporation_date,
        licenseIssuedDate: c.license_issued_date || undefined,
        lastRenewalDate: c.last_renewal_date || undefined,
        renewalDate: c.renewal_date,
        status: c.status,
        isLoanEligible: !!c.is_loan_eligible,
        vatNumber: c.vat_number || undefined,
        vatRegistrationDate: c.vat_registration_date || undefined,
        bankName: c.bank_name || undefined,
        activeProducts: parseJsonArray(c.active_products_json),
        linkedLeadId: c.linked_lead_id || undefined,
        contacts,
        documents,
        communicationPreferences: comm,
    });
});
router.post('/', authMiddleware_1.requireAuth, async (req, res) => {
    try {
        const id = await (0, companiesRepo_1.createCompany)(req.body);
        res.status(201).json({ id });
    }
    catch (err) {
        console.error(err);
        if (err.code === 'ER_DUP_ENTRY') {
            return res.status(409).json({ message: 'Company with this Trade License already exists.' });
        }
        res.status(500).json({ message: 'Failed to create company' });
    }
});
router.put('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    const ok = await (0, companiesRepo_1.updateCompany)(req.params.uuid, req.body);
    if (!ok)
        return res.status(404).json({ message: 'Not found' });
    res.json({ ok: true });
});
router.delete('/:uuid', authMiddleware_1.requireAuth, async (req, res) => {
    const ok = await (0, companiesRepo_1.deleteCompany)(req.params.uuid);
    if (!ok)
        return res.status(404).json({ message: 'Not found' });
    res.json({ ok: true });
});
router.post('/:uuid/contacts', authMiddleware_1.requireAuth, async (req, res) => {
    const contactId = await (0, companiesRepo_1.addContact)(req.params.uuid, req.body);
    if (!contactId)
        return res.status(404).json({ message: 'Company not found' });
    res.status(201).json({ id: String(contactId) });
});
router.put('/:uuid/contacts/:contactId', authMiddleware_1.requireAuth, async (req, res) => {
    const ok = await (0, companiesRepo_1.updateContact)(req.params.uuid, parseInt(req.params.contactId, 10), req.body);
    if (!ok)
        return res.status(404).json({ message: 'Not found' });
    res.json({ ok: true });
});
router.delete('/:uuid/contacts/:contactId', authMiddleware_1.requireAuth, async (req, res) => {
    const ok = await (0, companiesRepo_1.deleteContact)(req.params.uuid, parseInt(req.params.contactId, 10));
    if (!ok)
        return res.status(404).json({ message: 'Not found' });
    res.json({ ok: true });
});
router.patch('/:uuid/loan-eligibility', authMiddleware_1.requireAuth, async (req, res) => {
    const eligible = !!req.body.isLoanEligible;
    const ok = await (0, companiesRepo_1.setLoanEligibility)(req.params.uuid, eligible);
    if (!ok)
        return res.status(404).json({ message: 'Not found' });
    // TODO: create notification here linking to user context
    res.json({ ok: true });
});
router.patch('/:uuid/renewal-date', authMiddleware_1.requireAuth, async (req, res) => {
    const ok = await (0, companiesRepo_1.setRenewalDate)(req.params.uuid, req.body.renewalDate);
    if (!ok)
        return res.status(404).json({ message: 'Not found' });
    // TODO: optionally create a renewal task
    res.json({ ok: true });
});
router.post('/:uuid/documents', authMiddleware_1.requireAuth, upload_1.upload.single('file'), async (req, res) => {
    if (!req.file) {
        return res.status(400).json({ message: 'No file uploaded' });
    }
    const extension = req.file.originalname.split('.').pop()?.toUpperCase() || 'FILE';
    const documentData = {
        name: req.body.name || req.file.originalname,
        type: extension,
        url: `/uploads/${req.file.filename}`
    };
    const docId = await (0, companiesRepo_1.addDocument)(req.params.uuid, documentData);
    if (!docId)
        return res.status(404).json({ message: 'Company not found' });
    res.status(201).json({ id: String(docId) });
});
exports.default = router;
