"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const authMiddleware_1 = require("../middleware/authMiddleware");
const db_1 = require("../config/db");
const notificationsRepo_1 = require("../repositories/notificationsRepo");
const router = (0, express_1.Router)();
// Simple daily automation: set loan eligibility and create renewal tasks
router.post('/run-daily', authMiddleware_1.requireAuth, async (_req, res) => {
    // Loan eligibility: incorporation_date > 365 days
    const [companies] = await db_1.pool.query(`SELECT id, uuid, name, incorporation_date, is_loan_eligible FROM companies`);
    const today = new Date();
    let loanUpdates = 0;
    for (const c of companies) {
        const incDate = new Date(c.incorporation_date);
        const diffDays = Math.floor((today.getTime() - incDate.getTime()) / (1000 * 60 * 60 * 24));
        if (diffDays > 365 && !c.is_loan_eligible) {
            await db_1.pool.query('UPDATE companies SET is_loan_eligible=1 WHERE id=?', [c.id]);
            loanUpdates++;
            // For demo, notify the first admin found
            const [users] = await db_1.pool.query("SELECT id FROM users WHERE role IN ('Admin','Sales Manager','Super Admin') ORDER BY role LIMIT 1");
            if (users[0])
                await (0, notificationsRepo_1.createNotification)(users[0].id, 'New Loan Opportunity', `Company ${c.name} is now loan eligible`, 'info', `/companies/${c.uuid}`);
        }
    }
    // Renewal task creation: within next 30 days and no open task exists
    const [dueSoon] = await db_1.pool.query(`SELECT c.id, c.uuid, c.name, c.renewal_date FROM companies c WHERE c.renewal_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)`);
    let renewalTasksCreated = 0;
    for (const c of dueSoon) {
        const [openTasks] = await db_1.pool.query(`SELECT id FROM tasks WHERE type='Renewal' AND status IN ('Pending','In Progress') AND related_company_id=?`, [c.id]);
        if (!openTasks.length) {
            // assign to a manager/admin
            const [users] = await db_1.pool.query("SELECT id FROM users WHERE role IN ('Sales Manager','Admin','Super Admin') ORDER BY role LIMIT 1");
            const ownerId = users[0]?.id;
            if (ownerId) {
                await db_1.pool.query('INSERT INTO tasks (uuid, title, description, due_date, priority, status, type, owner_id, related_company_id, related_entity_display) VALUES (UUID(),?,?,?,?,?,?,?,?,?)', [`License Renewal Due - ${c.name}`, 'Auto-created by automation', c.renewal_date, 'High', 'Pending', 'Renewal', ownerId, c.id, c.name]);
                renewalTasksCreated++;
            }
        }
    }
    res.json({ loanUpdates, renewalTasksCreated });
});
exports.default = router;
