"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.findUserByEmail = findUserByEmail;
exports.findUserById = findUserById;
exports.listUsers = listUsers;
exports.countUsers = countUsers;
exports.createUser = createUser;
exports.updateUser = updateUser;
exports.deleteUser = deleteUser;
const db_1 = require("../config/db");
const uuid_1 = require("uuid");
const bcrypt_1 = __importDefault(require("bcrypt"));
async function findUserByEmail(email) {
    const [rows] = await db_1.pool.query('SELECT * FROM users WHERE email = ?', [email]);
    return rows[0] || null;
}
async function findUserById(id) {
    const [rows] = await db_1.pool.query('SELECT * FROM users WHERE id = ?', [id]);
    return rows[0] || null;
}
async function listUsers() {
    const [rows] = await db_1.pool.query('SELECT id, uuid, name, email, role, is_active, created_at, updated_at FROM users');
    return rows;
}
async function countUsers() {
    const [rows] = await db_1.pool.query('SELECT COUNT(*) as c FROM users');
    const first = rows[0];
    return Number(first?.c ?? 0);
}
async function createUser(user) {
    const { name, email, password, role } = user;
    const uuid = (0, uuid_1.v4)();
    const hashedPassword = await bcrypt_1.default.hash(password, 10);
    await db_1.pool.query('INSERT INTO users (uuid, name, email, password_hash, role) VALUES (?, ?, ?, ?, ?)', [uuid, name, email, hashedPassword, role]);
    return { uuid };
}
async function updateUser(uuid, updates) {
    const { name, email, role, password } = updates;
    const fields = [];
    const values = [];
    if (name) {
        fields.push('name = ?');
        values.push(name);
    }
    if (email) {
        fields.push('email = ?');
        values.push(email);
    }
    if (role) {
        fields.push('role = ?');
        values.push(role);
    }
    if (password) {
        const hashedPassword = await bcrypt_1.default.hash(password, 10);
        fields.push('password_hash = ?');
        values.push(hashedPassword);
    }
    if (fields.length === 0)
        return false;
    values.push(uuid);
    const [result] = await db_1.pool.query(`UPDATE users SET ${fields.join(', ')} WHERE uuid = ?`, values);
    return result.affectedRows > 0;
}
async function deleteUser(uuid) {
    const [result] = await db_1.pool.query('DELETE FROM users WHERE uuid = ?', [uuid]);
    return result.affectedRows > 0;
}
