"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.listLeads = listLeads;
exports.getLeadByUuid = getLeadByUuid;
exports.createLead = createLead;
exports.updateLead = updateLead;
exports.addLeadActivity = addLeadActivity;
const db_1 = require("../config/db");
const uuid_1 = require("uuid");
async function listLeads(filters, currentUser) {
    const where = [];
    const params = [];
    if (filters.status) {
        where.push('status = ?');
        params.push(filters.status);
    }
    if (filters.service) {
        where.push('service = ?');
        params.push(filters.service);
    }
    if (filters.ownerId) {
        where.push('owner_id = ?');
        params.push(filters.ownerId);
    }
    if (filters.source) {
        where.push('source = ?');
        params.push(filters.source);
    }
    if (filters.search) {
        where.push('(name LIKE ? OR company_name LIKE ? OR email LIKE ?)');
        params.push(`%${filters.search}%`, `%${filters.search}%`, `%${filters.search}%`);
    }
    if (filters.createdFrom) {
        where.push('created_at >= ?');
        params.push(filters.createdFrom);
    }
    if (filters.createdTo) {
        where.push('created_at <= ?');
        params.push(filters.createdTo);
    }
    // Sales Executive visibility restriction
    if (currentUser.role === 'Sales Executive') {
        where.push('owner_id = ?');
        params.push(currentUser.id);
    }
    const sql = `SELECT l.id, l.uuid, l.name, l.company_name, l.email, l.phone, l.service, l.status, l.owner_id, u.name AS owner_name, l.source, l.preferred_contact_method, l.value, l.notes, l.company_open_date, l.tags_json, l.created_at, l.updated_at FROM leads l LEFT JOIN users u ON l.owner_id = u.id` + (where.length ? ` WHERE ${where.join(' AND ')}` : '') + ' ORDER BY l.created_at DESC LIMIT 200';
    const [rows] = await db_1.pool.query(sql, params);
    return rows;
}
async function getLeadByUuid(uuid) {
    const [rows] = await db_1.pool.query('SELECT l.*, u.name AS owner_name FROM leads l LEFT JOIN users u ON l.owner_id=u.id WHERE l.uuid = ? LIMIT 1', [uuid]);
    const lead = rows[0];
    if (!lead)
        return null;
    const [acts] = await db_1.pool.query('SELECT * FROM lead_activities WHERE lead_id = ? ORDER BY activity_date DESC', [lead.id]);
    return { lead, activities: acts };
}
async function createLead(leadData, ownerId) {
    const uuid = (0, uuid_1.v4)();
    const sql = `INSERT INTO leads (uuid, name, company_name, email, phone, service, status, owner_id, source, preferred_contact_method, value, notes, company_open_date, tags_json) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;
    const params = [
        uuid,
        leadData.name,
        leadData.companyName,
        leadData.email,
        leadData.phone,
        leadData.service,
        'New',
        ownerId,
        leadData.source,
        leadData.preferredContactMethod,
        leadData.value || 0,
        leadData.notes,
        leadData.companyOpenDate,
        JSON.stringify(leadData.tags || [])
    ];
    const [res] = await db_1.pool.query(sql, params);
    // Add initial activity
    const leadId = res.insertId;
    await db_1.pool.query('INSERT INTO lead_activities (lead_id, type, content, activity_date, user_id) VALUES (?, ?, ?, NOW(), ?)', [leadId, 'StatusChange', 'Lead created', ownerId]);
    return { id: leadId, uuid };
}
async function updateLead(uuid, updates) {
    const fields = [];
    const params = [];
    if (updates.status) {
        fields.push('status = ?');
        params.push(updates.status);
    }
    if (updates.ownerId) {
        // Resolve UUID to ID
        const [uRows] = await db_1.pool.query('SELECT id FROM users WHERE uuid = ?', [updates.ownerId]);
        const user = uRows[0];
        if (user) {
            fields.push('owner_id = ?');
            params.push(user.id);
        }
    }
    if (fields.length === 0)
        return false;
    params.push(uuid);
    const sql = `UPDATE leads SET ${fields.join(', ')} WHERE uuid = ?`;
    await db_1.pool.query(sql, params);
    return true;
}
async function addLeadActivity(leadUuid, activity, userId) {
    // Get lead ID first
    const [rows] = await db_1.pool.query('SELECT id FROM leads WHERE uuid = ?', [leadUuid]);
    const lead = rows[0];
    if (!lead)
        return;
    const sql = `INSERT INTO lead_activities (lead_id, type, content, activity_date, user_id) VALUES (?, ?, ?, NOW(), ?)`;
    await db_1.pool.query(sql, [lead.id, activity.type, activity.content, userId]);
}
