"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.listCompanies = listCompanies;
exports.getCompanyByUuid = getCompanyByUuid;
exports.createCompany = createCompany;
exports.updateCompany = updateCompany;
exports.deleteCompany = deleteCompany;
exports.addContact = addContact;
exports.updateContact = updateContact;
exports.deleteContact = deleteContact;
exports.setLoanEligibility = setLoanEligibility;
exports.setRenewalDate = setRenewalDate;
exports.addDocument = addDocument;
const db_1 = require("../config/db");
async function listCompanies(filters) {
    const where = [];
    const params = [];
    if (filters.status) {
        where.push('status = ?');
        params.push(filters.status);
    }
    if (filters.isLoanEligible !== undefined) {
        where.push('is_loan_eligible = ?');
        params.push(filters.isLoanEligible ? 1 : 0);
    }
    if (filters.renewalFrom) {
        where.push('renewal_date >= ?');
        params.push(filters.renewalFrom);
    }
    if (filters.renewalTo) {
        where.push('renewal_date <= ?');
        params.push(filters.renewalTo);
    }
    if (filters.search) {
        where.push('(name LIKE ? OR trade_license LIKE ? OR vat_number LIKE ?)');
        params.push(`%${filters.search}%`, `%${filters.search}%`, `%${filters.search}%`);
    }
    const sql = `SELECT * FROM companies` + (where.length ? ` WHERE ${where.join(' AND ')}` : '') + ' ORDER BY renewal_date ASC LIMIT 200';
    const [rows] = await db_1.pool.query(sql, params);
    const companies = rows;
    for (const c of companies) {
        const [contacts] = await db_1.pool.query('SELECT * FROM company_contacts WHERE company_id = ?', [c.id]);
        const [docs] = await db_1.pool.query('SELECT * FROM company_documents WHERE company_id = ?', [c.id]);
        c.contacts = contacts;
        c.documents = docs;
    }
    return companies;
}
async function getCompanyByUuid(uuid) {
    const [rows] = await db_1.pool.query('SELECT * FROM companies WHERE uuid = ? LIMIT 1', [uuid]);
    const c = rows[0];
    if (!c)
        return null;
    const [contacts] = await db_1.pool.query('SELECT * FROM company_contacts WHERE company_id = ?', [c.id]);
    const [prefs] = await db_1.pool.query('SELECT * FROM company_communication_preferences WHERE company_id = ?', [c.id]);
    const [docs] = await db_1.pool.query('SELECT * FROM company_documents WHERE company_id = ?', [c.id]);
    return { company: c, contacts: contacts, preferences: prefs[0] || null, documents: docs };
}
async function createCompany(payload) {
    const [res] = await db_1.pool.query('INSERT INTO companies (uuid, name, trade_license, incorporation_date, license_issued_date, last_renewal_date, renewal_date, status, is_loan_eligible, vat_number, vat_registration_date, bank_name, active_products_json, linked_lead_id) VALUES (UUID(),?,?,?,?,?,?,?,?,?,?,?,?,?)', [payload.name, payload.tradeLicense, payload.incorporationDate, payload.licenseIssuedDate || null, payload.lastRenewalDate || null, payload.renewalDate, payload.status, payload.isLoanEligible ? 1 : 0, payload.vatNumber || null, payload.vatRegistrationDate || null, payload.bankName || null, JSON.stringify(payload.activeProducts || []), payload.linkedLeadId || null]);
    const insertId = res.insertId;
    const [rows] = await db_1.pool.query('SELECT uuid FROM companies WHERE id = ?', [insertId]);
    return rows[0].uuid;
}
async function updateCompany(uuid, payload) {
    const [res] = await db_1.pool.query('UPDATE companies SET name=?, trade_license=?, incorporation_date=?, license_issued_date=?, last_renewal_date=?, renewal_date=?, status=?, is_loan_eligible=?, vat_number=?, vat_registration_date=?, bank_name=?, active_products_json=?, linked_lead_id=? WHERE uuid=?', [payload.name, payload.tradeLicense, payload.incorporationDate, payload.licenseIssuedDate || null, payload.lastRenewalDate || null, payload.renewalDate, payload.status, payload.isLoanEligible ? 1 : 0, payload.vatNumber || null, payload.vatRegistrationDate || null, payload.bankName || null, JSON.stringify(payload.activeProducts || []), payload.linkedLeadId || null, uuid]);
    return res.affectedRows > 0;
}
async function deleteCompany(uuid) {
    const [rows] = await db_1.pool.query('SELECT id FROM companies WHERE uuid=?', [uuid]);
    const c = rows[0];
    if (!c)
        return false;
    const [res] = await db_1.pool.query('DELETE FROM companies WHERE id=?', [c.id]);
    return res.affectedRows > 0;
}
async function addContact(uuid, contact) {
    const [rows] = await db_1.pool.query('SELECT id FROM companies WHERE uuid=?', [uuid]);
    const c = rows[0];
    if (!c)
        return null;
    const [res] = await db_1.pool.query('INSERT INTO company_contacts (company_id, name, role, email, phone, is_primary) VALUES (?,?,?,?,?,?)', [c.id, contact.name, contact.role || null, contact.email || null, contact.phone || null, contact.isPrimary ? 1 : 0]);
    return res.insertId;
}
async function updateContact(uuid, contactId, contact) {
    const [rows] = await db_1.pool.query('SELECT id FROM companies WHERE uuid=?', [uuid]);
    const c = rows[0];
    if (!c)
        return false;
    const [res] = await db_1.pool.query('UPDATE company_contacts SET name=?, role=?, email=?, phone=?, is_primary=? WHERE id=? AND company_id=?', [contact.name, contact.role || null, contact.email || null, contact.phone || null, contact.isPrimary ? 1 : 0, contactId, c.id]);
    return res.affectedRows > 0;
}
async function deleteContact(uuid, contactId) {
    const [rows] = await db_1.pool.query('SELECT id FROM companies WHERE uuid=?', [uuid]);
    const c = rows[0];
    if (!c)
        return false;
    const [res] = await db_1.pool.query('DELETE FROM company_contacts WHERE id=? AND company_id=?', [contactId, c.id]);
    return res.affectedRows > 0;
}
async function setLoanEligibility(uuid, eligible) {
    const [res] = await db_1.pool.query('UPDATE companies SET is_loan_eligible=? WHERE uuid=?', [eligible ? 1 : 0, uuid]);
    return res.affectedRows > 0;
}
async function setRenewalDate(uuid, renewalDate) {
    const [res] = await db_1.pool.query('UPDATE companies SET renewal_date=? WHERE uuid=?', [renewalDate, uuid]);
    return res.affectedRows > 0;
}
async function addDocument(uuid, document) {
    const [rows] = await db_1.pool.query('SELECT id FROM companies WHERE uuid=?', [uuid]);
    const c = rows[0];
    if (!c)
        return null;
    const [res] = await db_1.pool.query('INSERT INTO company_documents (company_id, name, url, type, uploaded_at) VALUES (?,?,?,?, NOW())', [c.id, document.name, document.url, document.type || 'Document']);
    return res.insertId;
}
